/**
 * Content Script: Digital Codes Visit Tracker
 * 
 * Automatically tracks user visits to ICC Digital Codes pages and stores
 * them for quick access in the extension popup. Runs only on matching pages.
 * 
 * Pattern: https://codes.iccsafe.org/content/[CODE_ID]
 * Example: https://codes.iccsafe.org/content/OHPC2024P1
 */
(function() {
    const url = window.location.href;
    
    // Extract code identifier from URL path (e.g., "OHPC2024P1" from full URL)
    const match = url.match(/^https:\/\/codes\.iccsafe\.org\/content\/([^\/\?]+)/);
    
    if (match) {
        const codeBase = match[1];
        const pageTitle = document.title;
        
        // Only track valid code pages (not generic landing pages or sub-sections)
        if (codeBase && pageTitle && pageTitle !== 'ICC Digital Codes') {
            // Create visit record
            const recentPage = {
                codeBase: codeBase,
                title: pageTitle,
                url: `https://codes.iccsafe.org/content/${codeBase}`,
                timestamp: Date.now()
            };
            
            // Update stored visit history
            chrome.storage.local.get(['recentDigitalCodes'], function(result) {
                let recentPages = result.recentDigitalCodes || [];
                
                // Remove duplicate entries for this code (updates timestamp)
                recentPages = recentPages.filter(page => page.codeBase !== codeBase);
                
                // Add current visit to front of list
                recentPages.unshift(recentPage);
                
                // Maintain maximum of 10 recent entries
                recentPages = recentPages.slice(0, 10);
                
                // Save updated list back to storage
                chrome.storage.local.set({recentDigitalCodes: recentPages});
            });
        }
    }
})();
